// $Id: CXYPad.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CXYPad.hpp"
using Exponent::GUI::Controls::CXYPad;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CXYPad, CControl);

//	===========================================================================
CXYPad::CXYPad(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener) 
	  : CControl(root, uniqueId, area, listener)
	  , m_state(CRolloverButton::e_mouseOff)
	  , m_isDragging(false)
	  , m_verticalRange(0)
	  , m_horizontalRange(0)
	  , m_xValue(1.0)
	  , m_yValue(0.0)
	  , m_handleImage(NULL)
	  , m_handleRolloverImage(NULL)
	  , m_handleDownImage(NULL)
	  , m_handleColour(CAlphaColour::CALPHACOLOUR_PURPLE)
	  , m_handleOverColour(CAlphaColour::CALPHACOLOUR_ORANGE)
	  , m_handleDownColour(CAlphaColour::CALPHACOLOUR_RED)
{
	EXPONENT_CLASS_CONSTRUCTION(CXYPad);

	m_handleArea.setRect(0, 0, m_area.getWidth() / 10, m_area.getHeight() / 10);
	m_verticalRange   = (double)(m_area.getHeight() - m_handleArea.getHeight());
	m_horizontalRange = (double)(m_area.getWidth()  - m_handleArea.getWidth());
}

//	===========================================================================
CXYPad::~CXYPad()
{
	EXPONENT_CLASS_DESTRUCTION(CXYPad);
	FORGET_COUNTED_OBJECT(m_handleImage);
	FORGET_COUNTED_OBJECT(m_handleRolloverImage);
	FORGET_COUNTED_OBJECT(m_handleDownImage);
}

//	===========================================================================
void CXYPad::handleLeftButtonDown(CMouseEvent &event)
{
	// We lock the control to this
	m_rootControl->lockControl(this);

	if (event.isShiftDown())
	{
		// Lock to vertical
		m_handleArea.setTop(event.getMousePosition().getYPosition() - (m_handleArea.getHeight() / 2));
	}
	else if (event.isCtrlDown())
	{
		// Lock to horizontal
		m_handleArea.setLeft(event.getMousePosition().getXPosition() - (m_handleArea.getWidth() / 2));
	}
	else 
	{
		// Move the handle to the mouse position
		m_handleArea.setLeft(event.getMousePosition().getXPosition() - (m_handleArea.getWidth() / 2));
		m_handleArea.setTop(event.getMousePosition().getYPosition() - (m_handleArea.getHeight() / 2));
	}

	// Range check all the rectangles
	if (m_handleArea.getRight() > m_area.getWidth())
	{
		m_handleArea.setLeft(m_handleArea.getLeft() - (m_handleArea.getRight() - m_area.getWidth()));
	}
	if (m_handleArea.getBottom() > m_area.getHeight())
	{
		m_handleArea.setTop(m_handleArea.getTop() - (m_handleArea.getBottom() - m_area.getHeight()));
	}
	if (m_handleArea.getTop() < 0)
	{
		m_handleArea.setTop(0);
	}
	if (m_handleArea.getLeft() < 0)
	{
		m_handleArea.setLeft(0);
	}

	// Set the grab offset
	m_grabOffset = event.getMousePosition();
	m_grabOffset.offset(CPoint(-m_handleArea.getLeft(), -m_handleArea.getTop()));

	// We are now dragging the control
	m_isDragging = true;

	// Change the state
	m_state = CRolloverButton::e_mouseDown;

	// Store the value
	m_yValue = 1.0 - ((double)(m_handleArea.getTop()) / m_verticalRange);
	m_xValue = (double)(m_handleArea.getLeft()) / m_horizontalRange;

	// Notify the listener
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this));
	}

	// Redraw the window
	this->update();
}

//	===========================================================================
void CXYPad::handleLeftButtonUp(CMouseEvent &event)
{
	if (!m_handleArea.pointIsInside(event.getMousePosition()))
	{
		m_rootControl->unlockControl();
		m_state = CRolloverButton::e_mouseOff;
	}
	else
	{
		m_state = CRolloverButton::e_mouseOver;
	}
	m_isDragging = false;
	m_grabOffset.setPoint(0, 0);
	event.getMutableMouse()->setCursor();
	this->update();
}

//	===========================================================================
void CXYPad::handleDoubleClick(CMouseEvent &event)
{
	this->setXValue(0.5);
	this->setYValue(0.5);
	this->update();
}

//	===========================================================================
void CXYPad::handleMouseMovement(CMouseEvent &event)
{
	if (m_isDragging)
	{
		if (event.isShiftDown())
		{
			// Lock to vertical
			const long yPos = event.getMousePosition().getYPosition();

			// Set the top
			m_handleArea.setTop(CBounds::ensureRange(yPos - m_grabOffset.getYPosition(), 0, m_area.getHeight() - m_handleArea.getHeight()));

			// Store the value
			m_yValue = 1.0 - ((double)(m_handleArea.getTop()) / m_verticalRange);

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this));
			}
		}
		else if (event.isCtrlDown())
		{
			// Lock to horizontal
			const long xPos = event.getMousePosition().getXPosition();

			// Set the top
			m_handleArea.setLeft(CBounds::ensureRange(xPos - m_grabOffset.getXPosition(), 0, m_area.getWidth() - m_handleArea.getWidth()));

			// Store the value
			m_xValue = (double)(m_handleArea.getLeft()) / m_horizontalRange;

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this));
			}
		}
		else
		{
			// Lock to vertical
			const long yPos = event.getMousePosition().getYPosition();
			const long xPos = event.getMousePosition().getXPosition();

			// Set the top
			m_handleArea.setLeft(CBounds::ensureRange(xPos - m_grabOffset.getXPosition(), 0, m_area.getWidth() - m_handleArea.getWidth()));
			m_handleArea.setTop(CBounds::ensureRange(yPos - m_grabOffset.getYPosition(), 0, m_area.getHeight() - m_handleArea.getHeight()));

			// Store the value
			m_yValue = 1.0 - ((double)(m_handleArea.getTop()) / m_verticalRange);
			m_xValue = (double)(m_handleArea.getLeft()) / m_horizontalRange;

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this));
			}
		}
	}
	else
	{
		this->checkAndLock(event.getMousePosition());
	}
	this->update();
}

//	===========================================================================
void CXYPad::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	// Now draw the hnadle
	graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
	IImage *theImage = NULL;
	CAlphaColour theColour;
	switch(m_state)
	{
		case CRolloverButton::e_mouseOff:
			if (m_handleImage)
			{
				theImage = m_handleImage;
			}
			else
			{
				theImage  = NULL;
				theColour = m_handleColour;
			}
		break;
		case CRolloverButton::e_mouseOver:
			if (m_handleRolloverImage)
			{
				theImage = m_handleRolloverImage;
			}
			else if (m_handleImage)
			{
				theImage = m_handleImage;
			}
			else
			{
				theImage  = NULL;
				theColour = m_handleOverColour;
			}
		break;
		case CRolloverButton::e_mouseDown:
			if (m_handleDownImage)
			{
				theImage = m_handleDownImage;
			}
			else if (m_handleImage)
			{
				theImage = m_handleImage;
			}
			else
			{
				theImage  = NULL;
				theColour = m_handleDownColour;
			}
		break;
	}
	if (theImage)
	{
		graphics.drawImage(theImage, m_handleArea, theImage->getNormalisedImageSize());
	}
	else
	{
		graphics.getMutableBrush()->setColour(theColour);
		graphics.fillRectangle(m_handleArea);
		graphics.drawRectangle(m_handleArea);
	}
}

//	===========================================================================
void CXYPad::setXValue(const double xValue)
{
	if (xValue >= 0.0 && xValue <= 1.0)
	{
		m_xValue = xValue;
		m_handleArea.setLeft((long)(m_xValue * m_horizontalRange));
		this->update();
	}
}

//	===========================================================================
void CXYPad::setYValue(const double yValue)
{
	if (yValue >= 0.0 && yValue <= 1.0)
	{
		m_yValue = yValue;
		m_handleArea.setTop((long)((1.0 - m_yValue) * m_verticalRange));
		this->update();
	}
}

//	===========================================================================
void CXYPad::setHandleImage(IImage *handleImage, IImage *rolloverHandleImage, IImage *downHandleImage)
{
	EXCHANGE_COUNTED_OBJECTS(m_handleImage, handleImage);
	EXCHANGE_COUNTED_OBJECTS(m_handleRolloverImage, rolloverHandleImage);
	EXCHANGE_COUNTED_OBJECTS(m_handleDownImage, downHandleImage);

	if (m_handleImage)
	{
		m_handleArea.setWidth(m_handleImage->getImageSize().getWidth());
		m_handleArea.setHeight(m_handleImage->getImageSize().getHeight());

		m_verticalRange   = (double)(m_area.getHeight() - m_handleArea.getHeight());
		m_horizontalRange = (double)(m_area.getWidth()  - m_handleArea.getWidth());
	}
}

//	===========================================================================
void CXYPad::checkAndLock(const CPoint &point)
{
	if (m_handleArea.pointIsInside(point))
	{
		m_state = CRolloverButton::e_mouseOver;
		m_rootControl->lockControl(this);
	}
	else
	{
		m_state = CRolloverButton::e_mouseOff;
		m_rootControl->unlockControl();
	}
}

//	===========================================================================
void CXYPad::setColours(CAlphaColour frameColour, CAlphaColour backgroundColour, CAlphaColour handleColour, CAlphaColour handleOverColour, CAlphaColour handleDownColour)
{
	this->setDefaultControlColours(backgroundColour, frameColour);
	m_handleColour	   = handleColour;
	m_handleOverColour = handleOverColour;
	m_handleDownColour = handleDownColour;
}